<?php
/*
 * pppoe_ha_event.php — CARP event handler + staged MASTER recovery + periodic reconcile
 *
 * Usage from devd:
 *   /usr/local/sbin/pppoe_ha_event carp <subsystem> <MASTER|BACKUP|INIT>
 *     - <subsystem> may be "5@igb0", "carp1", or just "5"
 *
 * Manual invocations:
 *   /usr/local/sbin/pppoe_ha_event reconcile
 *   /usr/local/sbin/pppoe_ha_event reconcile_quiet
 *
 * What this script does
 * ---------------------
 *  - Maps CARP VHID events to actions on target interfaces (PPPoE WAN).
 *  - MASTER path (two-stage, CARP-suppressed):
 *      0) Immediately bring PPPoE real iface up (if present) or pfSctl reload friendly iface.
 *      1) Suppress CARP events.
 *      2) After 30s (still MASTER?) -> targeted reconcile (NO pfSctl reload here anymore).
 *      3) After another 30s -> safety targeted reconcile again.
 *      4) Clear suppression.
 *  - BACKUP: bring PPPoE down.
 *  - INIT: do nothing unless a fresh CARP read says state is actually BACKUP (avoid flapping).
 *  - Periodic quiet reconcile every 5 minutes (single background loop with a pidfile).
 *
 * Notes
 * -----
 *  - pfSctl must be called with the FRIENDLY interface name (e.g. 'wan').
 *  - ifconfig up/down must be called on the REAL interface (e.g. 'pppoe0').
 *  - During the MASTER stabilization window we ignore CARP events.
 */

require_once("util.inc");
require_once("config.inc");
require_once("interfaces.inc");
require_once("/usr/local/pkg/pppoe_ha.inc");

/* ---------------- System log helpers ---------------- */

function ha_log($msg, $prio = LOG_NOTICE) {
    static $opened = false;
    if (!$opened) {
        openlog('pppoe-ha', LOG_PID, LOG_USER);
        $opened = true;
        register_shutdown_function(function() { closelog(); });
    }
    syslog($prio, $msg);
}
function ha_log_debug($msg) { ha_log($msg, LOG_DEBUG); }

/* ---------------- Files/paths ---------------- */

const SUPPRESS_FILE = '/var/run/pppoe_ha.suppress.json';      // CARP hold-off window
const PERIODIC_PID  = '/var/run/pppoe_ha_periodic.pid';       // 5 min reconcile loop

/* ---------------- Utility: sanitize tokens ---------------- */

function ppha_sanitize_token($s) {
    $s = trim((string)$s);
    if ($s !== '' && $s[0] === '$') { $s = substr($s, 1); }
    return $s;
}

/* ---------------- Build target list from package config ---------------- */

function ppha_build_targets($only_vhid = null) {
    $rows = ppha_get_rows();
    $vips = config_get_path('virtualip/vip', []);
    $targets = [];

    if (!is_array($rows) || !is_array($vips)) { return []; }

    foreach ($rows as $i => $row) {
        $enabled = isset($row['enabled']) && strcasecmp((string)$row['enabled'], 'ON') === 0;
        $vipref  = array_key_exists('vipref', $row) ? (string)$row['vipref'] : '';
        $iface   = array_key_exists('iface',  $row) ? (string)$row['iface']  : '';
        if (!$enabled || $vipref === '' || $iface === '') {
            ha_log_debug("row[$i] skipped (enabled/vipref/iface missing or off)");
            continue;
        }
        $vip = $vips[$vipref] ?? null;
        if (!$vip || ($vip['mode'] ?? '') !== 'carp') {
            ha_log_debug("row[$i] skipped (vipref {$vipref} not CARP or missing)");
            continue;
        }
        $vhid = (int)($vip['vhid'] ?? -1);
        if ($vhid < 0) {
            ha_log_debug("row[$i] skipped (vipref {$vipref} has invalid VHID)");
            continue;
        }
        if ($only_vhid !== null && (int)$only_vhid !== $vhid) { continue; }

        $real = get_real_interface($iface);
        if (empty($real)) {
            ha_log("row[$i] {$iface}/vipref={$vipref} - real interface not found; skipping");
            continue;
        }

        $targets[] = [
            'idx'            => (int)$i,
            'iface_friendly' => $iface,   // 'wan'
            'iface_real'     => $real,    // 'pppoe0'
            'vipref'         => $vipref,
            'vhid'           => $vhid,
        ];
    }
    return $targets;
}

/* ---------------- CARP parsing & state helpers ---------------- */

function parse_carp_subsystem($subsys) {
    $subsys = ppha_sanitize_token($subsys);
    if (preg_match('/^(\d+)\@([A-Za-z0-9_.:\-]+)$/', $subsys, $m)) {
        return ['vhid'=>(int)$m[1], 'real'=>$m[2], 'carpif'=>null];
    }
    if (preg_match('/^(carp\d+)$/', $subsys, $m)) {
        $carpif = $m[1];
        $out=[]; @exec("/sbin/ifconfig ".escapeshellarg($carpif)." 2>/dev/null",$out);
        $vhid=null; foreach ($out as $line) {
            if (preg_match('/\bvhid\s+(\d+)/', $line, $mm)) { $vhid=(int)$mm[1]; break; }
        }
        return ['vhid'=>$vhid, 'real'=>null, 'carpif'=>$carpif];
    }
    if (preg_match('/^\d+$/', $subsys)) { return ['vhid'=>(int)$subsys, 'real'=>null, 'carpif'=>null]; }
    return ['vhid'=>null, 'real'=>null, 'carpif'=>null];
}

/** Read CARP state (MASTER|BACKUP|INIT) for a given VHID. */
function get_carp_state_for_vhid($vhid) {
    $vhid = (int)$vhid;
    $out = []; $rc = 0;
    @exec('/sbin/ifconfig -a', $out, $rc);
    if ($rc !== 0 || empty($out)) { return null; }
    foreach ($out as $line) {
        if (preg_match('/\bcarp:\s*(MASTER|BACKUP|INIT)\b.*\bvhid\s+(\d+)/i', $line, $m)) {
            if ((int)$m[2] === $vhid) { return strtoupper($m[1]); }
        }
    }
    return null;
}

/* ---------------- PPPoE helpers ---------------- */

function is_pppoe_real($ifname){ return (bool)preg_match('/^pppoe\d+$/', (string)$ifname); }

function real_iface_present($real){
    $out = []; $rc = 0;
    @exec("/sbin/ifconfig " . escapeshellarg($real) . " 2>/dev/null", $out, $rc);
    return ($rc === 0 && !empty($out));
}

function get_pppoe_status($real) {
    $out = [];
    @exec("/sbin/ifconfig " . escapeshellarg($real) . " 2>&1", $out);
    $has_v4_p2p = false; $has_v6_global = false;
    foreach ($out as $line) {
        if (preg_match('/^\s*inet\s+\S+\s+-->\s+\S+/', $line)) { $has_v4_p2p = true; continue; }
        if (preg_match('/^\s*inet6\s+([0-9a-f:]+)/i', $line, $m)) {
            $addr = strtolower($m[1]);
            if (strpos($addr, 'fe80:') !== 0 && stripos($line, 'tentative') === false) { $has_v6_global = true; }
        }
    }
    return ['active'=>($has_v4_p2p || $has_v6_global), 'has_ipv4_p2p'=>$has_v4_p2p, 'has_v6_global'=>$has_v6_global];
}

/* pfSense-friendly up/down:
 *  - iface_up(): pfSctl 'interface reload <friendly>'
 *  - iface_down(): /sbin/ifconfig <real> down
 */
function iface_up($friendly){
    $cmd = "/usr/local/sbin/pfSctl -c " . escapeshellarg("interface reload {$friendly}");
    mwexec($cmd);
}
function iface_down($real){
    mwexec("/sbin/ifconfig " . escapeshellarg($real) . " down");
}

/* ---------------- CARP suppression window ---------------- */

function ppha_set_suppression(int $seconds, string $reason, int $vhid = 0) {
    $until = time() + max(1, $seconds);
    $data = ['until'=>$until, 'reason'=>$reason, 'vhid'=>$vhid];
    @file_put_contents(SUPPRESS_FILE, json_encode($data));
    ha_log("suppress CARP events for {$seconds}s (reason={$reason}, vhid={$vhid})");
}
function ppha_clear_suppression() {
    if (file_exists(SUPPRESS_FILE)) { @unlink(SUPPRESS_FILE); ha_log("suppression cleared"); }
}
function ppha_is_suppressed() {
    if (!file_exists(SUPPRESS_FILE)) return false;
    $j = json_decode(@file_get_contents(SUPPRESS_FILE), true);
    if (!is_array($j) || empty($j['until'])) return false;
    if (time() >= (int)$j['until']) { @unlink(SUPPRESS_FILE); return false; }
    return true;
}

/* ---------------- Background spawner ---------------- */

function spawn_bg(array $args) {
    $php  = PHP_BINARY ?: "/usr/local/bin/php";
    $self = escapeshellarg(realpath(__FILE__));
    $cmd  = $php . " -f " . $self . ' ' . implode(' ', array_map('escapeshellarg', $args)) . " >/dev/null 2>&1 &";
    mwexec($cmd);
}

/* ---------------- Periodic quiet reconcile (every 5 minutes) ---------------- */

function periodic_ensure_running() {
    if (file_exists(PERIODIC_PID)) {
        $pid = (int)trim(@file_get_contents(PERIODIC_PID));
        if ($pid > 0) {
            $out=[]; $rc=0;
            @exec('ps -p '.escapeshellarg((string)$pid).' -o pid=', $out, $rc);
            if ($rc === 0 && !empty($out)) { return; }
            @unlink(PERIODIC_PID);
        }
    }
    spawn_bg(['PERIODIC']);
}

function periodic_loop() {
    $pid = getmypid();
    @file_put_contents(PERIODIC_PID, (string)$pid);
    ha_log("periodic reconcile loop started (pid={$pid})", LOG_INFO);

    while (true) {
        if (!file_exists(PERIODIC_PID)) { break; }
        if (!ppha_is_suppressed()) {
            reconcile_all(true); // quiet
        } else {
            ha_log_debug("periodic: suppression active, skip reconcile");
        }
        sleep(300);
    }
    ha_log("periodic reconcile loop stopped", LOG_INFO);
    exit(0);
}

/* ---------------- Core action: apply target state ---------------- */

function ppha_apply_target_state(array $t, string $state) {
    $iface = $t['iface_friendly']; // 'wan'
    $real  = $t['iface_real'];     // 'pppoe0'
    $vhid  = $t['vhid'];

    if (!is_pppoe_real($real)) {
        ha_log("warning: {$real} does not look like pppoeX; continuing anyway");
    }

    switch ($state) {
        case 'MASTER':
            ha_log("VHID {$vhid} MASTER - staged UP {$iface} ({$real})");
            // Stage 0: immediate bring-up or reload
            if (real_iface_present($real)) {
                mwexec("/sbin/ifconfig " . escapeshellarg($real) . " up");
                ha_log("pppoe immediate up on {$real}");
            } else {
                ha_log("pppoe real iface {$real} absent -> pfSctl reload {$iface}");
                iface_up($iface);
            }
            // Suppress CARP while we settle (now 90s is enough: 30s wait + 30s wait + buffer)
            ppha_set_suppression(90, 'master_stabilize', $vhid);
            // Background post-master sequence (NO pfSctl reload step anymore)
            spawn_bg(['MASTER_POST', (string)$vhid, $iface, $real]);
            break;

        case 'BACKUP':
            ha_log("VHID {$vhid} BACKUP - DOWN {$iface} ({$real})");
            iface_down($real);
            break;

        case 'INIT':
            // INIT pulses often happen during pfsync/promisc churn; only down if we are *really* BACKUP now.
            $now = get_carp_state_for_vhid($vhid);
            if ($now === 'BACKUP') {
                ha_log("VHID {$vhid} INIT (current BACKUP) - DOWN {$iface} ({$real})");
                iface_down($real);
            } else {
                ha_log("VHID {$vhid} INIT - no action (current={$now})");
            }
            break;

        default:
            ha_log("VHID {$vhid} {$state} - no action");
    }
}

/* ---------------- Targeted reconcile for one VHID ---------------- */

function reconcile_target(int $vhid, bool $quiet=false) {
    $targets = ppha_build_targets($vhid);
    if (!$targets) {
        if (!$quiet) ha_log("Reconcile target: no mappings for VHID {$vhid}");
        return;
    }
    foreach ($targets as $t) {
        $iface = $t['iface_friendly']; $real = $t['iface_real'];
        $state = get_carp_state_for_vhid($vhid) ?? 'INIT';

        if ($state === 'INIT') {
            if (!$quiet) ha_log("Reconcile target: VHID {$vhid} INIT - skip");
            continue;
        }

        if ($state === 'MASTER') {
            if (is_pppoe_real($real)) {
                $st = get_pppoe_status($real);
                if ($st['active']) {
                    if (!$quiet) ha_log("Reconcile target: MASTER & PPPoE active on {$real} - done");
                    continue;
                }
            }
            if (!$quiet) ha_log("Reconcile target: MASTER - ensuring WAN ready (pfSctl reload if needed)");
            iface_up($iface); // harmless if already OK; pfSense will no-op quickly
        } else { // BACKUP
            if (!$quiet) ha_log("Reconcile target: BACKUP - bringing {$real} down");
            iface_down($real);
        }
    }
}

/* ---------------- Full reconcile across all mappings ---------------- */

function reconcile_all(bool $quiet=false) {
    $targets = ppha_build_targets();
    if (!$targets) { if (!$quiet) ha_log("Reconcile: no mappings configured"); return; }
    if (!$quiet) { ha_log("Reconcile: evaluating " . count($targets) . " mapping(s)"); }

    foreach ($targets as $t) {
        $vhid = $t['vhid'];
        $state = get_carp_state_for_vhid($vhid) ?? 'INIT';

        if ($state === 'INIT') {
            if (!$quiet) ha_log("Reconcile: VHID {$vhid} state=INIT - skip");
            continue;
        }

        if ($state === 'MASTER') {
            if (is_pppoe_real($t['iface_real'])) {
                $st = get_pppoe_status($t['iface_real']);
                if ($st['active']) {
                    if (!$quiet) ha_log("Reconcile: VHID {$vhid} MASTER and PPPoE already UP - skip");
                    continue;
                }
            }
            if (!$quiet) ha_log("Reconcile: VHID {$vhid} MASTER - pfSctl reload {$t['iface_friendly']}");
            iface_up($t['iface_friendly']);
        } else { // BACKUP
            if (!$quiet) ha_log("Reconcile: VHID {$vhid} BACKUP - down {$t['iface_real']}");
            iface_down($t['iface_real']);
        }
    }
}

/* ---------------- MASTER post sequence (background) ---------------- */

function master_post_sequence(int $vhid, string $iface, string $real) {
    // Stage A: wait 30s, confirm still MASTER, then targeted reconcile (NO pfSctl reload step before this).
    sleep(30);
    $cur = get_carp_state_for_vhid($vhid);
    if ($cur !== 'MASTER') {
        ha_log("master_post: aborted — VHID {$vhid} no longer MASTER ({$cur})");
        ppha_clear_suppression();
        return;
    }
    ha_log("master_post: still MASTER — targeted reconcile VHID {$vhid}");
    reconcile_target($vhid, false);

    // Stage B: wait another 30s, do a safety targeted reconcile again.
    sleep(30);
    $cur = get_carp_state_for_vhid($vhid);
    if ($cur === 'MASTER') {
        ha_log("master_post: safety targeted reconcile VHID {$vhid}");
        reconcile_target($vhid, true); // quiet safety pass
    } else {
        ha_log("master_post: skipped safety reconcile — state became {$cur}");
    }

    // Done — clear suppression
    ppha_clear_suppression();
}

/* ---------------- Event entrypoints ---------------- */

function handle_carp_state_change($vhid, $state) {
    if (ppha_is_suppressed()) {
        ha_log("CARP event suppressed: VHID {$vhid} state={$state}");
        return;
    }
    $targets = ppha_build_targets($vhid);
    if (!$targets) { ha_log("no mappings for VHID {$vhid}; ignoring"); return; }
    foreach ($targets as $t) { ppha_apply_target_state($t, $state); }
}

function main_entry($argv) {
    $argv0 = array_shift($argv);
    $cmd   = strtoupper($argv[0] ?? '');

    // Keep the periodic loop alive
    periodic_ensure_running();

    if ($cmd === 'CARP') {
        $subsys = ppha_sanitize_token($argv[1] ?? '');
        $state  = strtoupper(ppha_sanitize_token($argv[2] ?? ''));
        $info   = parse_carp_subsystem($subsys);
        $vhid   = $info['vhid'];

        ha_log("Handle CARP command for {$subsys} - {$state}");
        if ($vhid === null || !in_array($state, ['MASTER','BACKUP','INIT'], true)) {
            ha_log("Invalid CARP args: subsystem={$subsys} parsed_vhid=".var_export($vhid,true)." state={$state}");
            exit(1);
        }
        handle_carp_state_change($vhid, $state);
        exit(0);
    }

    if ($cmd === 'RECONCILE') { reconcile_all(false); exit(0); }
    if ($cmd === 'RECONCILE_QUIET') { reconcile_all(true); exit(0); }

    if ($cmd === 'MASTER_POST') {
        $vhid = (int)($argv[1] ?? 0);
        $iface = (string)($argv[2] ?? '');
        $real  = (string)($argv[3] ?? '');
        if ($vhid <= 0 || $iface === '' || $real === '') { exit(1); }
        master_post_sequence($vhid, $iface, $real);
        exit(0);
    }

    if ($cmd === 'PERIODIC') { periodic_loop(); exit(0); }

    /* help */
    echo "Usage:\n";
    echo "  pppoe_ha_event carp <vhid|carpX|vhid@realif> <MASTER|BACKUP|INIT>\n";
    echo "  pppoe_ha_event reconcile\n";
    echo "  pppoe_ha_event reconcile_quiet\n";
    exit(1);
}

/* ---- run ---- */
main_entry($argv);
